/*jshint esversion: 6 */
/*global define, console */

define(["lib/Zoot", "src/utils", "src/math/mathUtils", "src/math/Vec2", "lib/dev", "lib/tasks"],
function (Zoot, utils, mathUtils, v2, dev, tasks) {

	"use strict";

	function canIterpolateHeadLabel (inHeadLabel) {
		return (inHeadLabel !== "Head/MouthShape");
	}	
	
	var head14Labels = { // note: values are identity; repeated in .cpp code in FaceMetrics constructor (TODO: factor)
			"Head/DX" : 0,
			"Head/DY" : 0,
			"Head/DZ" : 0,
			"Head/Orient/X" : 0,
			"Head/Orient/Y" : 0,
			"Head/Orient/Z" : 0,
			"Head/LeftEyebrow" : 1,
			"Head/RightEyebrow" : 1,
			"Head/LeftEyelid" : 1,
			"Head/RightEyelid" : 1,
			"Head/MouthSX" : 1,
			"Head/MouthSY" : 1,
			"Head/MouthDX" : 0,
			"Head/MouthDY" : 0,
			"Head/Scale" : 1,
			"Head/MouthShape" : -1,
			"Head/LeftEyeGazeX" : 0,
			"Head/LeftEyeGazeY" : 0, 
			"Head/RightEyeGazeX" : 0, 
			"Head/RightEyeGazeY" : 0
		},

		// see https://bitbucket.org/amitibo/pyfacetracker/src/d54866d9b3e23654b1c06adca625dafcbe7629ce/doc/images/3DMonaLisa.png?at=default
		//	for diagram of these indices
//		faceFeatureIndices = {
//			"Head" :					[0, 16], // jaw line going from puppet right to left
//			"Head/Right Eye" :			[36, 41],
//			"Head/Left Eye" :			 [42, 47],
//			"Head/Right Eyebrow" :		 [17, 21],
//			"Head/Left Eyebrow" :		 [22, 26],
//			"Head/Nose" :				 [27, 35], // 27-30 down ridge, 31-35 right to left
//			"Head/Mouth" :				 [48, 65]
//		},


//		note: mouths are in UI order
// 		hard-coded, instead of sorting by UI name, so the order remains the same across UI languages
 
		mouthShapeLayerTagDefinitions = [
			{
				id: "Adobe.Face.NeutralMouth",	// see parallel definition in FaceTracker
				artMatches: ["neutral"],
				uiName: "$$$/animal/Behavior/Face/TagName/Neutral=Neutral",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Mouth", subSort:1}]				
			},
			{
				id: "Adobe.Face.SmileMouth",
				artMatches: ["smile"],
				uiName: "$$$/animal/Behavior/Face/TagName/Smile=Smile",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Mouth", subSort:1}]				
			},
			{
				id: "Adobe.Face.SurprisedMouth",
				artMatches: ["surprised"],
				uiName: "$$$/animal/Behavior/Face/TagName/Surprised=Surprised",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Mouth", subSort:1}]				
			},
		],
		
		mouthParentLayerTagDefinition = [
			{
				id: "Adobe.Face.MouthsParent",
				artMatches: ["mouth"],
				uiName: "$$$/animal/Behavior/Face/TagName/MouthGroup=Mouth Group",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Face"}]				
			},
		],
		
		viewLayerTagDefinitions = [
			{
				id: "Adobe.Face.LeftProfile",
				artMatches: ["left profile"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftProfile=Left Profile",
			},
			
			{
				id: "Adobe.Face.LeftQuarter",
				artMatches: ["left quarter"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftQuarter=Left Quarter",
			},
			
			{
				id: "Adobe.Face.Front",
				artMatches: ["frontal"],
				uiName: "$$$/animal/Behavior/Face/TagName/Frontal=Frontal",
			},
			
			{
				id: "Adobe.Face.RightQuarter",
				artMatches: ["right quarter"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightQuarter=Right Quarter",
			},
			
			{
				id: "Adobe.Face.RightProfile",
				artMatches: ["right profile"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightProfile=Right Profile",
			},
			
			{
				id: "Adobe.Face.Upward",
				artMatches: ["upward"],
				uiName: "$$$/animal/Behavior/Face/TagName/Upward=Upward",
			},

			{
				id: "Adobe.Face.Downward",
				artMatches: ["downward"],
				uiName: "$$$/animal/Behavior/Face/TagName/Downward=Downward",
			},
			
		];
		
		viewLayerTagDefinitions.forEach (function (tagDefn) {
			tagDefn.tagType = "layertag";
			tagDefn.uiGroups = [{ id:"Adobe.TagGroup.HeadTurn"}];			
		});
		
		var miscLayerTagDefinitions = [
			{
				id: "Adobe.Face.LeftBlink",
				artMatches: ["left blink"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftBlink=Left Blink",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Face"}]				

			},
			{
				id: "Adobe.Face.RightBlink",
				artMatches: ["right blink"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightBlink=Right Blink",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Face"}]								
			},
		],
		
		// all possible face features presented as tag definitions, also used in defineHandleParams()
		eyelidLayerTagDefinitions = [
			{
				id: "Adobe.Face.LeftEyelidTopLayer",
				artMatches: ["left eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidTopSize=Left Eyelid Top Size",
			},
		
			{
				id: "Adobe.Face.RightEyelidTopLayer",
				artMatches: ["right eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidTopSize=Right Eyelid Top Size",
			},
		
			{
				id: "Adobe.Face.LeftEyelidBottomLayer",
				artMatches: ["left eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidBottomSize=Left Eyelid Bottom Size",
			},
		
			{
				id: "Adobe.Face.RightEyelidBottomLayer",
				artMatches: ["right eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidBottomSize=Right Eyelid Bottom Size",
			}
		],
		// all possible face features presented as tag definitions, also used in defineHandleParams()
		pupilLayerTagDefinitions = [
			{
				id: "Adobe.Face.LeftPupilRange",
				artMatches: ["left eyeball", "left pupil range"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftPupilRange=Left Pupil Range",
			},
		
			{
				id: "Adobe.Face.RightPupilRange",
				artMatches: ["right eyeball", "right pupil range"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightPupilRange=Right Pupil Range",
			},
			{
				id: "Adobe.Face.LeftPupilSize",
				artMatches: ["left pupil", "left pupil size"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftPupilSize=Left Pupil Size",
			},
		
			{
				id: "Adobe.Face.RightPupilSize",
				artMatches: ["right pupil", "right pupil size"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightPupilSize=Right Pupil Size",
			},
		],
		headFeatureTagDefinitions = [	
		
			{
				id: "Adobe.Face.Head",
				artMatches: ["head"],
				uiName: "$$$/animal/Behavior/Face/TagName/Head=Head",
			},
		],
		eyeFeatureTagDefinitions = [	
			{
				id: "Adobe.Face.LeftEye",
				artMatches: ["left eye"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEye=Left Eye",
			},
			{
				id: "Adobe.Face.RightEye",
				artMatches: ["right eye"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEye=Right Eye",
			}
		],
		faceFeatureTagDefinitions = [	
			{
				id: "Adobe.Face.LeftEyelidTop",
				artMatches: ["left eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidTop=Left Eyelid Top",
			},
		
			{
				id: "Adobe.Face.RightEyelidTop",
				artMatches: ["right eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidTop=Right Eyelid Top",
			},
		
			{
				id: "Adobe.Face.LeftEyelidBottom",
				artMatches: ["left eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidBottom=Left Eyelid Bottom",
			},

			{
				id: "Adobe.Face.RightEyelidBottom",
				artMatches: ["right eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidBottom=Right Eyelid Bottom",
			},
		
			//"Adobe.Face.Right Blink",		don't need these because there are no transforms related to them (just replacements)
			//"Adobe.Face.Left Blink",

			{
				id: "Adobe.Face.LeftEyebrow",
				artMatches: ["left eyebrow"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyebrow=Left Eyebrow",
			},
		
			{
				id: "Adobe.Face.RightEyebrow",
				artMatches: ["right eyebrow"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyebrow=Right Eyebrow",
			},
		
			{
				id: "Adobe.Face.Nose",
				artMatches: ["nose"],
				uiName: "$$$/animal/Behavior/Face/TagName/Nose=Nose",
			},
		
			{
				id: "Adobe.Face.Mouth",
				artMatches: ["mouth"],
				uiName: "$$$/animal/Behavior/Face/TagName/Mouth=Mouth",
			}
		],
		pupilFeatureTagDefinitions = [	
		
			{
				id: "Adobe.Face.LeftPupil",
				artMatches: ["left pupil"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftPupil=Left Pupil",
			},

			{
				id: "Adobe.Face.RightPupil",
				artMatches: ["right pupil"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightPupil=Right Pupil",
			}
		
		],
		// stores mapping from head14 params to puppet transformations	
			// all transformation values are unitless; when transforms are applied
			// at runtime, they get multiplied by the appropriate puppet-specific measurements
			// note: no mappings for Left/Right Blink because it's only hidden or shown
		head14ToPuppetTransformMapping = { 
			"Adobe.Face.Head" :
			[
			{
				"labels" : ["Head/DX"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/DX" : -2 
					},
					"T" : {
						"translate" : [-2, 0], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/DX" : 2
					},
					"T" : {
						"translate" : [2, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			},
			{
				"labels" : ["Head/DY"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/DY" : -2 
					},
					"T" : {
						"translate" : [0, -2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/DY" : 2
					},
					"T" : {
						"translate" : [0, 2],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			},
			{
				"labels" : ["Head/Orient/Z"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/Orient/Z" : -1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1],
						"angle" : -1.5
					}
				},
				{
					"head14" : {
						"Head/Orient/Z" : 1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1], 
						"angle" : 1.5
					}
				}
				]
			}			
			],

			"Adobe.Face.Nose" : 
			[
			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "noseDepth", 
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]);
					T.translate[1] += 0.5 * Math.sin(x[1]); 
				}
			}
			],

			"Adobe.Face.LeftEyebrow" :
			[
			{
				"labels" : ["Head/LeftEyebrow"],
				"translateUnit" : "leftEyeEyebrowDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/LeftEyebrow" : 0.5
					},
					"T" : {
						"translate" : [0, 2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/LeftEyebrow" : 2
					},
					"T" : {
						"translate" : [0, -4], 
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}

			],

			"Adobe.Face.RightEyebrow" :
			[
			{
				"labels" : ["Head/RightEyebrow"],
				"translateUnit" : "rightEyeEyebrowDist",
				"samples" : [

				{
					"head14" : {
						"Head/RightEyebrow" : 0.5
					},
					"T" : {
						"translate" : [0, 2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyebrow" : 2
					},
					"T" : {
						"translate" : [0, -4],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}

			],

			"Adobe.Face.RightEye" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1.2], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}
			],

			"Adobe.Face.RightEyelidTop" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"translateUnit" : "rightEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.RightEyelidBottom" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"translateUnit" : "rightEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.RightPupil" :
			[
			{
				"labels" : ["Head/RightEyeGazeX"],
				"translateUnit" : "rightEyeGazeRangeX",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyeGazeX" : -1.0
					},
					"T" : {
						"translate" : [-1.0, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyeGazeX" : 1.0
					},
					"T" : {
						"translate" : [1.0, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/RightEyeGazeY"],
				"translateUnit" : "rightEyeGazeRangeY",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyeGazeY" : -1.0
					},
					"T" : {
						"translate" : [0, -1.0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyeGazeY" : 1.0
					},
					"T" : {
						"translate" : [0, 1.0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.LeftEye" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1.2], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}
			],

			"Adobe.Face.LeftEyelidTop" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"translateUnit" : "leftEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.LeftEyelidBottom" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"translateUnit" : "leftEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			}
			],

			"Adobe.Face.LeftPupil" :
			[
			{
				"labels" : ["Head/LeftEyeGazeX"],
				"translateUnit" : "leftEyeGazeRangeX",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyeGazeX" : -1.0
					},
					"T" : {
						"translate" : [-1.0, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyeGazeX" : 1.0
					},
					"T" : {
						"translate" : [1.0, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/LeftEyeGazeY"],
				"translateUnit" : "leftEyeGazeRangeY",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyeGazeY" : -1.0
					},
					"T" : {
						"translate" : [0, -1.0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyeGazeY" : 1.0
					},
					"T" : {
						"translate" : [0, 1.0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.Mouth" :
			[
			{
				"labels" : ["Head/MouthDX"],
				"translateUnit" : "mouthWidth",				
				"samples" : [

				{
					"head14" : {
						"Head/MouthDX" : -1
					},
					"T" : {
						"translate" : [-1, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthDX" : 1
					},
					"T" : {
						"translate" : [1, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				}

				]
			},
			{
				"labels" : ["Head/MouthDY"],
				"translateUnit" : "mouthHeight",				
				"samples" : [

				{
					"head14" : {
						"Head/MouthDY" : -1
					},
					"T" : {
						"translate" : [0, -1],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthDY" : 1
					},
					"T" : {
						"translate" : [0, 1],
						"scale" : [1, 1],
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/MouthSX"],
				"samples" : [

				{
					"head14" : {
						"Head/MouthSX" : 0
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [0, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthSX" : 1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1.5, 1],
						"angle" : 0
					}
				}

				]
			},
			{
				"labels" : ["Head/MouthSY"],
				"samples" : [

				{
					"head14" : {
						"Head/MouthSY" : 0
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthSY" : 5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 5],
						"angle" : 0
					}
				}
				]
			}, 
			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}			
			]
		},
		cameraInputPauseKeyCodeEnglish = Zoot.keyCodes.getKeyGraphId(";"),
		cameraInputPauseKeyCodeGerman = Zoot.keyCodes.getKeyGraphId(","),
		cameraInputPauseKeyCodeV2 = Zoot.keyCodes.getKeyGraphId(Zoot.nml.translateZstring("$$$/animal/Behavior/FaceUtils/cameraInputPauseKeyCommand/OnlySemicolonOrCommaSupported=;")),
		cameraInputParameterDefinitionV2 = {	id: "cameraInput", type: "eventGraph", 
											uiName: "$$$/animal/Behavior/FaceTracker/Parameter/cameraInput=Camera Input",
											inputKeysArray: ["Head/", cameraInputPauseKeyCodeEnglish, cameraInputPauseKeyCodeGerman],
											outputKeyTraits: {
												takeGroupsArray: [
													{
														id: "Filtered/Head/*"
													}
												]
											},
											disarmIfAllInputKeysDisabledArray: ["Head/"], // Used to determine whether to disarm if dependent hardware is disabled.
											uiToolTip: "$$$/animal/Behavior/FaceTracker/param/cameraInput/tooltip=Analyzed face data from the camera; pause the face pose by holding down semicolon (;)",
											defaultArmedForRecordOn: true, supportsBlending: true};

	// all possible face features presented as tag definitions, also used in defineHandleParams()
	headFeatureTagDefinitions.forEach ( function (tagDefn) {
		tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];							
		tagDefn.tagType = "handletag";
	});
	eyeFeatureTagDefinitions.forEach( function (tagDefn) {
		tagDefn.tagType = "handletag";
		tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];					
	});
	faceFeatureTagDefinitions.forEach( function (tagDefn) {
		tagDefn.tagType = "handletag";
		tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];					
	});
	pupilFeatureTagDefinitions.forEach( function (tagDefn) {
		tagDefn.tagType = "handletag";
		tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];					
	});
	eyelidLayerTagDefinitions.forEach ( function (tagDefn) {
		tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];							
		tagDefn.tagType = "layertag";
	});
	pupilLayerTagDefinitions.forEach ( function (tagDefn) {
		tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];							
		tagDefn.tagType = "layertag";
	});
	miscLayerTagDefinitions.forEach ( function (tagDefn) {
		tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];							
		tagDefn.tagType = "layertag";
	});
	
	function makeValidIdFromLabel (str) {
		return str.replace(/[^\w]/g, "_");
	}
	
	function makeLayerIdFromLabel (str) {
		return "L_" + makeValidIdFromLabel(str);
	}

	function makeHandleIdFromLabel (str) {
		return "H_" + makeValidIdFromLabel(str);
	}
	
	/* unused
	function setTransformTranslate(transforms, transformName, translate) {
		var t = transforms[transformName];
		if (t) {
			t.translate = translate;
		}
	}*/

	function addHiddenLayerParam (aParams, aMatchIn, label, tooltip) {
		var firstMatch = aMatchIn[0];
		var aMatches = [];
		aMatchIn.forEach(function (match) {
			aMatches.push("//" + match);
		});
		
		aParams.push({id:makeLayerIdFromLabel(firstMatch), type:"layer", uiName:label,
					dephault:{match:aMatches, startMatchingAtParam: "viewLayers"},
					maxCount:1, uiToolTip:tooltip, hidden:true});
	}
	
	function addLayerParamAtIndex (aParams, aMatchIn, label, tooltip, index) {
		var firstMatch = aMatchIn[0];
		var aMatches = [];
		var param = {id:makeLayerIdFromLabel(firstMatch), type:"layer", uiName:label,
					dephault:{match:aMatches, startMatchingAtParam: "viewLayers"},
					maxCount:1, uiToolTip:tooltip, hidden:false};

		aMatchIn.forEach(function (match) {
			aMatches.push("//" + match);
		});
		
		aParams.splice(index, 0, param);
	}
	
	function defineHandleParams (includePupilsB, includeFaceB) {
		var aParams = [];
		
		utils.assert(includePupilsB || includeFaceB);

		headFeatureTagDefinitions.forEach( function (tagDefn) {
			var label = tagDefn.id;
			var def = {id:makeHandleIdFromLabel(label), type:"handle", uiName:tagDefn.uiName,
						dephault:{match:"//"+label},
						hidden:false};

			if (label !== "Adobe.Face.Head") {	// special case for head, as it is usually a parent of the views,
				def.maxCount = 1;				//	so we don't depend on it being inside a view (but do allow it)
				def.dephault.startMatchingAtParam = "viewLayers";
			}

			aParams.push(def);
		});
		
		eyeFeatureTagDefinitions.forEach( function (tagDefn) {
			var label = tagDefn.id;
			var def = {id:makeHandleIdFromLabel(label), type:"handle", uiName:tagDefn.uiName,
						dephault:{match:"//"+label},
						hidden:false};

			def.maxCount = 1;
			def.dephault.startMatchingAtParam = "viewLayers";

			aParams.push(def);
		});
		if (includeFaceB) {
			faceFeatureTagDefinitions.forEach( function (tagDefn) {
				var label = tagDefn.id;
				var def = {id:makeHandleIdFromLabel(label), type:"handle", uiName:tagDefn.uiName,
							dephault:{match:"//"+label},
							hidden:false};

				def.maxCount = 1;
				def.dephault.startMatchingAtParam = "viewLayers";

				aParams.push(def);
			});
		}

		if (includePupilsB) {
			pupilFeatureTagDefinitions.forEach( function (tagDefn) {
				var label = tagDefn.id;
				var def = {id:makeHandleIdFromLabel(label), type:"handle", uiName:tagDefn.uiName,
							dephault:{match:"//"+label},
							hidden:false};

				def.maxCount = 1;
				def.dephault.startMatchingAtParam = "viewLayers";

				aParams.push(def);
			});
		}

		var rightIndex, leftIndex;
		
		leftIndex = aParams.findIndex(function (element) { return element.id === makeHandleIdFromLabel(includePupilsB ? "Adobe.Face.RightPupil" : "Adobe.Face.RightEye"); });
		rightIndex = leftIndex + 1;

		if (includePupilsB) {
			addLayerParamAtIndex(aParams, ["Adobe.Face.LeftPupilRange"],
				"$$$/animal/behavior/face/tag/LeftPupilRange=Left Pupil Range",
				"$$$/animal/behavior/face/tag/LeftPupilRange/tooltip=Sets the movement range of the Left Pupil", leftIndex+1);

			addLayerParamAtIndex(aParams, ["Adobe.Face.RightPupilRange"], 
				"$$$/animal/behavior/face/tag/RightPupilRange=Right Pupil Range", 
				"$$$/animal/behavior/face/tag/RightPupilRange/tooltip=Sets the movement range of the Right Pupil", rightIndex+1);

			// and layer params for the pupils
			addHiddenLayerParam(aParams, ["Adobe.Face.LeftPupilSize"], 
				"Left Pupil", 
				"Used to compute range of the Left Pupil");
			addHiddenLayerParam(aParams, ["Adobe.Face.RightPupilSize"], 
				"Right Pupil", 
				"Used to compute range of the Right Pupil");
		}

		// and layer params for the eyelids (TODO: consolidate to only have layer params, not handles?)
		if (includeFaceB) {
			addHiddenLayerParam(aParams, ["Adobe.Face.LeftEyelidTopLayer"], 
				"Left Eyelid Top", 
				"Sets the vertical range of the Left Eyelid; if missing, Left Eyelid Top _handle_ is used instead");

			addHiddenLayerParam(aParams, ["Adobe.Face.RightEyelidTopLayer"], 
				"Right Eyelid Top", 
				"Sets the vertical range of the Right Eyelid; if missing, Right Eyelid Top _handle_ is used instead");

			addHiddenLayerParam(aParams, ["Adobe.Face.LeftEyelidBottomLayer"], 
				"Left Eyelid Bottom", 
				"Sets the vertical range of the Left Eyelid; if missing, Left Eyelid Bottom _handle_ is used instead");

			addHiddenLayerParam(aParams, ["Adobe.Face.RightEyelidBottomLayer"], 
				"Right Eyelid Bottom", 
				"Sets the vertical range of the Right Eyelid; if missing, Right Eyelid Bottom _handle_ is used instead");
		}

		return aParams;
	}
	
	function addTransformTranslate(transforms, transformName, translate) {
		var t = transforms[transformName];
		if (t) {
			t.translate[0] += translate[0];
			t.translate[1] += translate[1];
		}
	}

	// get named puppetMeasurement
	function getPuppetMeasurement(self, inMeasurementName, args, viewIndex) {
		var measurement = null, puppetMeasurements = self.aPuppetMeasurements[viewIndex];
		
		if (inMeasurementName && puppetMeasurements && puppetMeasurements[inMeasurementName]) {
			measurement = puppetMeasurements[inMeasurementName];

			// TODO: it's not ideal that some parameter adjustments happen here, while others
			// happen in computePuppetTransforms. should consolidate.
			// apply relevant parameter adjustments
			if (inMeasurementName === "noseDepth" || inMeasurementName === "eyeDepth") {
				measurement *= args.getParam("parallaxFactor") / 100;
			}
		}

		return measurement;
	}
	
	function setTransformScale(transforms, transformName, scale) {
		var t = transforms[transformName];
		if (t) {
			t.scale = scale;
		}
	}

	// get corresponding entry in head14ToPuppetTransformMapping
	function getHead14ToPuppetTransformEntry(inLabel) {
		var entry = null;
		if (head14ToPuppetTransformMapping.hasOwnProperty(inLabel)) {
			entry = head14ToPuppetTransformMapping[inLabel];
		}

		return entry;
	}

	function getIdentityTransform() {
		var identityTransform = {};
		identityTransform.translate = [0, 0];
		identityTransform.scale = [1, 1];
		identityTransform.angle = 0;
		return identityTransform;
	}

	function isEyebrowLabel(inLabel) {
		return inLabel.indexOf("Eyebrow") >= 0;
	}

	// given a set of head14 params and a mapping entry with list of samples
	// determine the appropriate transform to return
	function computePuppetTransform(self, inHead14, inFaceFeatureLabel, inMappingEntry, args, viewIndex) {
		var finalTransform = getIdentityTransform(), transform, labels, translateUnit, samples, custom, customArgs, 
		translateRange, scaleRange, minAngle, maxAngle, angleRange,
		i, j, v, label, inVec, vec1, vec2, range, offset, rangeNorm, 
		restOffset, restAlpha, alpha;

		for (i = 0; i < inMappingEntry.length; i += 1) {

			labels = inMappingEntry[i].labels;
			samples = inMappingEntry[i].samples;
			custom = inMappingEntry[i].custom;
			inVec = []; vec1 = []; vec2 = []; range = []; offset = [];
			translateRange = [0, 0];
			scaleRange = [0, 0];
			minAngle = 0;
			maxAngle = 0;
			restOffset = null;
			restAlpha = null;

			transform = getIdentityTransform();

			// interpolate based on mapping samples
			if (samples && samples.length >= 2) {

				// if computing eyebrow position, use user-specified up/down eyebrow angles
				if (isEyebrowLabel(inFaceFeatureLabel)) {
					minAngle = (inFaceFeatureLabel.indexOf("Left") >= 0) ? args.getParam("downEyebrowsTilt") : -args.getParam("downEyebrowsTilt");
					maxAngle = (inFaceFeatureLabel.indexOf("Left") >= 0) ? args.getParam("upEyebrowsTilt") : -args.getParam("upEyebrowsTilt");
					// the factor of 8 determines the default sensitivity of eyebrow rotation wrt eyebrow height
					minAngle = 7 * (minAngle/100) * (0.5*Math.PI);
					maxAngle = 7 * (maxAngle/100) * (0.5*Math.PI);
				} 
				else {
					minAngle = samples[0].T.angle;
					maxAngle = samples[samples.length-1].T.angle;
				}

				v2.subtract(samples[samples.length-1].T.translate, samples[0].T.translate, translateRange);
				v2.subtract(samples[samples.length-1].T.scale, samples[0].T.scale, scaleRange);
				angleRange = maxAngle - minAngle;

				for (j = 0; j < labels.length; j += 1) {
					label = labels[j];

					if (inHead14.hasOwnProperty(label)) {
						inVec.push(inHead14[label]);
					}

					if (samples[0].head14.hasOwnProperty(label)) {
						vec1.push(samples[0].head14[label]);
					}
					if (samples[samples.length-1].head14.hasOwnProperty(label)) {
						vec2.push(samples[samples.length-1].head14[label]);
					}
				}

				if (inVec.length === labels.length && vec1.length === labels.length && vec2.length === labels.length) {

					rangeNorm = 0;
					for (j = 0; j < labels.length; j +=1 ) {
						label = labels[j];
						range[j] = vec2[j] - vec1[j];
						offset[j] = inVec[j] - vec1[j];
						rangeNorm += (range[j] * range[j]);

						// compute restOffset for eyebrows (where rest head14 value is 1.0)
						if (isEyebrowLabel(label)) {
							restOffset = 1.0 - vec1[j];
						}
					}

					rangeNorm = Math.sqrt(rangeNorm);

					// get dot product of offset onto range
					alpha = 0;
					for (j = 0; j < labels.length; j += 1) {
						label = labels[j];
						alpha += offset[j] * (range[j] / rangeNorm);

						// compute restAlpha for eyebrows
						if (isEyebrowLabel(label)) {
							restAlpha = restOffset / range[j];
						}
					}
					alpha /= rangeNorm;

					// clamp
					if (alpha < 0) { alpha = 0; }
					if (alpha > 1) { alpha = 1; }

					// compute interpolated transform values
					v2.add(transform.translate, v2.add(samples[0].T.translate, v2.scale(alpha, translateRange, []), []), transform.translate);
					v2.xmy(transform.scale, v2.add(samples[0].T.scale, v2.scale(alpha, scaleRange, []), []), transform.scale);

					// if interpolating eyebrow angles, only use either the minAngle or maxAngle depending
					// on whether the eyebrows are above or below the rest position; this ensures that at the
					// rest pose, eyebrows are always level
					if (restAlpha) {
						if (alpha < restAlpha) {
							alpha = alpha / restAlpha;
							transform.angle += (1.0 - alpha) * minAngle;
						}
						else {
							alpha = (alpha - restAlpha) / (1.0 - restAlpha);
							transform.angle += (alpha * maxAngle);
						}
					}
					else {
						transform.angle += (minAngle + (alpha * angleRange));
					}
				}

			}
			else if (custom) {
				customArgs = [];
				for (j = 0; j < labels.length; j += 1) {
					label = labels[j];
					v = inHead14[label];
					if (v === undefined) {
						console.log("missing FaceTracker custom value for " + label);
					} else {
						customArgs.push(v);
					}
				}

				custom(customArgs, transform);
			}

			// get translateUnit and scale transform
			translateUnit = getPuppetMeasurement(self, inMappingEntry[i].translateUnit, args, viewIndex);

			if (translateUnit) {
				v2.scale(translateUnit, transform.translate, transform.translate);
			}

			// add to finalTransform
			v2.add(finalTransform.translate, transform.translate, finalTransform.translate);
			v2.xmy(finalTransform.scale, transform.scale, finalTransform.scale);
			finalTransform.angle += transform.angle;
		}

		return finalTransform;
	}

	// 1.0 means no change to the scale, 0.0 means scale will be identity (either 1.0 or -1.0), and 0.5
	//	means the scale will be half the strength (4 -> 2, 0.25 -> 0.5)
	function adjustScaleByFactor(scale, factor) {
		var negative, scaleDown;

		// handle 0.0 scale
		if (scale === 0) {
			if (factor > 0) {
				return scale;
			} else {
				return 1.0;
			}
		}

		if (factor === 1) {
			return scale;
		}
		
		negative = scale < 0;
		
		if (negative) {
			scale = -scale; // make it positive for a moment
		}
		
		scaleDown = scale < 1;
		
		if (scaleDown) {
			scale = 1/scale; // make it > 1 for a moment
		}
		
		// now we only need to deal with scale > 1 here
		scale = 1 + (scale - 1) * factor;

		if (scaleDown) {
			scale = 1/scale;
		}

		if (negative) {
			scale = -scale; // back to negative
		}
		
		return scale;
	}

	function adjustTransformByFactor(translate, factor) {
		return [translate[0] * factor, translate[1] * factor];
	}
	
	function applyFactorToTransform(inPosFactor, inScaleFactor, inRotFactor, inoutTransform) {
		// if getIdentityTransform() ever returns something other than actual identity, this function
		//	would need to be updated to do a LERP between that and the passed param
		inoutTransform.translate = adjustTransformByFactor(inoutTransform.translate, inPosFactor);
		inoutTransform.scale[0] = adjustScaleByFactor(inoutTransform.scale[0], inScaleFactor);
		inoutTransform.scale[1] = adjustScaleByFactor(inoutTransform.scale[1], inScaleFactor);
		inoutTransform.angle *= inRotFactor;
	}

	// customized version of applyParamFactorToNamedTransform that allows for adjusting pos, scale and rot factors.
	// the values in factors scale the effect of the param on the corresponding component of the transform.
	// e.g., factors.pos = 1 modifies translation by the param full param values, 
	//		and factors.pos = 0 does not modify translation at all
	function applyParamFactorToNamedTransformCustom(self, paramName, args, transforms, transformName, factors)
	{
		var t = transforms[transformName], factor;

		if (t) {
			factor = args.getParam(paramName) / 100;
			applyFactorToTransform(1 + (factor-1) * factors.pos, 1 + (factor-1) * factors.scale, 1 + (factor-1) * factors.rot, t);
		}		
	}

	// paramName is for a (currently root-level) param that has 100 as a "neutral" gain
	function applyParamFactorToNamedTransform(self, paramName, args, transforms, transformName) {
		var factors = { pos : 1, scale : 1, rot : 1 };
		applyParamFactorToNamedTransformCustom(self, paramName, args, transforms, transformName, factors);
	}	

	// compute transforms for puppet
	function getPuppetTransforms(self, args, head14, viewIndex) {
		var transforms = {}, transform,
			faceFeatureLabel, head14ToPuppetTransformEntry,
			featureLabels = self.aFeatureLabels[viewIndex]; 

		// apply parameter adjustments to puppet measurements
		for (faceFeatureLabel in featureLabels) {
			if (featureLabels.hasOwnProperty(faceFeatureLabel)) {
				head14ToPuppetTransformEntry = getHead14ToPuppetTransformEntry(faceFeatureLabel);
				if (featureLabels[faceFeatureLabel] && head14ToPuppetTransformEntry) {
					transform = computePuppetTransform(self, head14, faceFeatureLabel, head14ToPuppetTransformEntry, args, viewIndex);
					transforms[faceFeatureLabel] = transform;
				} 
			}
		}
		return transforms;
	}

	// compute transforms for puppet
	function computePuppetTransforms(self, args, head14, viewIndex) {
		var transforms = getPuppetTransforms(self, args, head14, viewIndex), headTransform,
			featureLabels = self.aFeatureLabels[viewIndex]; 

		// 
		// adjust eyes
		//
		if (featureLabels["Adobe.Face.RightEyelidTop"] && featureLabels["Adobe.Face.RightEyelidBottom"]) {
			setTransformScale(transforms, "Adobe.Face.RightEye", [1, 1]);
		} else {
			// not clear why this clause is needed, though harmless
			transforms["Adobe.Face.RightEyelidTop"] = getIdentityTransform();
			transforms["Adobe.Face.RightEyelidBottom"] = getIdentityTransform();
		}
		if (featureLabels["Adobe.Face.LeftEyelidTop"] && featureLabels["Adobe.Face.LeftEyelidBottom"]) {
			setTransformScale(transforms, "Adobe.Face.LeftEye", [1, 1]);
		} else {
			transforms["Adobe.Face.LeftEyelidTop"] = getIdentityTransform();
			transforms["Adobe.Face.LeftEyelidBottom"] = getIdentityTransform();
		}

		// when blink puppet is available, we don't want to scale the eye
		// TODO: this implies we should disable the eyeFactor param, but we currently don't have a 
		// way to do that (i.e. onCreateStageBehavior should return param UI hint that the param should be disabled)
		if (self.leftBlinkLayers[viewIndex].length > 0) {
			setTransformScale(transforms, "Adobe.Face.LeftEye", [1, 1]);
		} else {
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.LeftEye", { pos : 0, scale : 1, rot : 0 });
		}

		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.LeftEyelidTop", { pos : 1, scale : 0, rot : 0 });
		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.LeftEyelidBottom", { pos : 1, scale : 0, rot : 0 });			
		
		if (self.rightBlinkLayers[viewIndex].length > 0) {
			setTransformScale(transforms, "Adobe.Face.RightEye", [1, 1]);
		} else {
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.RightEye", { pos : 0, scale : 1, rot : 0 });
		}

		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.RightEyelidTop", { pos : 1, scale : 0, rot : 0 });
		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.RightEyelidBottom", { pos : 1, scale : 0, rot : 0 });			

		//
		// adjust eyebrows
		//
		applyParamFactorToNamedTransform(self, "eyebrowFactor", args, transforms, "Adobe.Face.LeftEyebrow");
		applyParamFactorToNamedTransform(self, "eyebrowFactor", args, transforms, "Adobe.Face.RightEyebrow");

		if (args.getParam("moveEyebrowsTogether")) {
			var leftEyebrowTransform = transforms["Adobe.Face.LeftEyebrow"],
				rightEyebrowTransform = transforms["Adobe.Face.RightEyebrow"],
				avgEyebrowTranslateY = 0, avgEyebrowAngle = 0;

			if (leftEyebrowTransform && rightEyebrowTransform) {
				avgEyebrowTranslateY = 0.5 * (leftEyebrowTransform.translate[1] + rightEyebrowTransform.translate[1]);
				avgEyebrowAngle = 0.5 * (leftEyebrowTransform.angle - rightEyebrowTransform.angle); // we subtract right eyebrow angle because it is mirrored

				leftEyebrowTransform.translate[1] = avgEyebrowTranslateY;
				leftEyebrowTransform.angle = avgEyebrowAngle;				
				rightEyebrowTransform.translate[1] = avgEyebrowTranslateY;
				rightEyebrowTransform.angle = -avgEyebrowAngle;
			}
		}

		//
		// adjust mouth
		//
		applyParamFactorToNamedTransform(self, "mouthFactor", args, transforms, "Adobe.Face.Mouth");

		//
		// adjust head
		//
		setTransformScale(transforms, "Adobe.Face.Head", [head14["Head/Scale"], head14["Head/Scale"]]);

		headTransform = transforms["Adobe.Face.Head"];
		if (headTransform) {
			applyFactorToTransform(
				args.getParam("headPosFactor") / 100, 
				args.getParam("headScaleFactor") / 100, 
				args.getParam("headRotFactor") / 100, 
				headTransform);
		}

		// DEBUG
		/* (uncomment var everyN far above)
		everyN += 1;
		if (everyN >= 1) {
			everyN = 0;
			// Put debugging output here
		}
		*/

		return transforms;
	}
	
	function computeEyeGazeRanges (self, args, viewIndex) {
		var puppetMeasurements = self.aPuppetMeasurements[viewIndex], 
			leftEyeGazeRangeX = 0, leftEyeGazeRangeY = 0, rightEyeGazeRangeX = 0, rightEyeGazeRangeY = 0,
			leftEyeWidth = getPuppetMeasurement(self, "leftEyeWidth", args, viewIndex),
			leftEyeHeight = getPuppetMeasurement(self, "leftEyeHeight", args, viewIndex),
			rightEyeWidth = getPuppetMeasurement(self, "rightEyeWidth", args, viewIndex),
			rightEyeHeight = getPuppetMeasurement(self, "rightEyeHeight", args, viewIndex),
			leftPupilWidth = getPuppetMeasurement(self, "leftPupilWidth", args, viewIndex),
			leftPupilHeight = getPuppetMeasurement(self, "leftPupilHeight", args, viewIndex),
			rightPupilWidth = getPuppetMeasurement(self, "rightPupilWidth", args, viewIndex),
			rightPupilHeight = getPuppetMeasurement(self, "rightPupilHeight", args, viewIndex);

		if (leftEyeWidth) {
			leftEyeGazeRangeX = (leftPupilWidth) ? 0.5 * (leftEyeWidth - leftPupilWidth) : 0.5 * leftEyeWidth;
		}
		if (leftEyeHeight) {
			leftEyeGazeRangeY = (leftPupilHeight) ? 0.5 * (leftEyeHeight - leftPupilHeight) : 0.5 * leftEyeHeight;
		}
		if (rightEyeWidth) {
			rightEyeGazeRangeX = (rightPupilWidth) ? 0.5 * (rightEyeWidth - rightPupilWidth) : 0.5 * rightEyeWidth;
		}
		if (rightEyeHeight) {
			rightEyeGazeRangeY = (rightPupilHeight) ? 0.5 * (rightEyeHeight - rightPupilHeight) : 0.5 * rightEyeHeight;
		}

		puppetMeasurements.leftEyeGazeRangeX = leftEyeGazeRangeX;
		puppetMeasurements.leftEyeGazeRangeY = leftEyeGazeRangeY;
		puppetMeasurements.rightEyeGazeRangeX = rightEyeGazeRangeX;
		puppetMeasurements.rightEyeGazeRangeY = rightEyeGazeRangeY;
	}
	
	// returns the first match from the layer param, or null
	function getLayer (args, label, viewIndex) {
		var match = args.getParam(makeLayerIdFromLabel(label))[viewIndex];
		
		if (match) {
			match = match[0];	// only get first element from array if there is one (might be an empty view)
		}
		
		if (!match) {
			match = null;					// the old code returned null
		}
		
		return match;
	}
	
	function getHandle (args, label, viewIndex) {
		var v = args.getParam(makeHandleIdFromLabel(label));
		
		if (label === "Adobe.Face.Head") {
			// special case: no startMatchingAtParam, but viewIndex is not the right way to span
			//	this array -- it's of unrelated length. Still, in practice,
			//	honoring it here means Head handles will actually move as long there are the
			//	same number or more views.
			//	TODO: find real fix for this, either by pulling access to this handle into
			//		a separate loop called once for each match, or figure out how to have Head
			//		not be a special view-less case (and still support the common practice of
			//		having multiple views inside a single Head group).
			return v[viewIndex];	// might be undefined
		} else {
			// only works for a single handle (per view) for each label right now
			utils.assert(Array.isArray(v));
			return v[viewIndex][0];	// might be undefined
		}
	}	
	
	function computePuppetMeasurements (self, args, viewIndex, includeFaceB, includePupilsB, includeMouthB) {
		function getLayerParam(id) {
			return getLayer(args, id, viewIndex);
		}
		function getHandleParam(id) {
			return getHandle(args, id, viewIndex);
		}
		
		function getHandleParentLayerFallback (id) {
			var layer = null, handle = getHandleParam(id);
			if (handle) {
				layer = handle.getPuppet().getParentLayer().getSdkLayer();
			}
			return layer;
		}

		function getBBox(layer) {
			var bbox = [0, 0, 0, 0], puppet, matPuppet, bboxPoints,
				ranges = { xmin:null, xmax:null, ymin:null, ymax:null };

			bbox = layer.getBounds();

			// getBounds returns the bbox wrt the layer's coordinate frame, 
			// but the layer itself may be transformed by a local matrix, 
			// so we scale/translate the bbox by the local matrix if found.
			// TODO: Is there a better way to check if the current layer has
			// its own local matrix? -wilmotli
			if (layer.getSource) {
				puppet = layer.getSource();
				if (puppet && puppet.getMatrix) {
					matPuppet = puppet.getMatrix();

					// NOTE: we compute axis-aligned bbox *after* transforming by the
					// local matrix; this matches implementation of Container::GetExtent
					bboxPoints = [
						[ bbox[0], bbox[1] ], 						// top left
						[ bbox[0] + bbox[2], bbox[1] ], 			// top right
						[ bbox[0], bbox[1] + bbox[3] ], 			// bottom left
						[ bbox[0] + bbox[2], bbox[1] + bbox[3] ]	// bottom right
					];

					bboxPoints = bboxPoints.map(function (pt) {
						return v2.transformAffine(matPuppet, pt, null);
					});

					bboxPoints.forEach(function (pt, ptIndex) {
						ranges.xmin = (ranges.xmin !== null) ? Math.min(ranges.xmin, pt[0]) : pt[0];
						ranges.xmax = (ranges.xmax !== null) ? Math.max(ranges.xmax, pt[0]) : pt[0];
						ranges.ymin = (ranges.ymin !== null) ? Math.min(ranges.ymin, pt[1]) : pt[1];
						ranges.ymax = (ranges.ymax !== null) ? Math.max(ranges.ymax, pt[1]) : pt[1];
					});

					bbox[0] = ranges.xmin;
					bbox[1] = ranges.ymin;
					bbox[2] = ranges.xmax - ranges.xmin;
					bbox[3] = ranges.ymax - ranges.ymin;
				}
			}

			return bbox;
		}

		// "pos" measures the x or y dist between two rest positions
		// "bbox" measures the x or y dimension of a rest bbox
		// "scaledCopy" copies another named measurement scaled by factor
		
		var measurementsList = [
				["headWidth", "bbox", self.headPuppet, "X"],
				["headHeight", "bbox", self.headPuppet, "Y"],
			
			],

			puppetMeasurements = {},
			i, j, value, m, bbox1, bbox2;
		
		
		if (includePupilsB) {
			measurementsList.push(["leftEyeWidth",	"bbox", getLayerParam("Adobe.Face.LeftPupilRange"), "X"]);
			measurementsList.push(["leftEyeWidth",	"bbox", getHandleParentLayerFallback("Adobe.Face.LeftEye"), "X"]);
			
			measurementsList.push(["rightEyeWidth",	"bbox", getLayerParam("Adobe.Face.RightPupilRange"), "X"]);
			measurementsList.push(["rightEyeWidth",	"bbox", getHandleParentLayerFallback("Adobe.Face.RightEye"), "X"]);
			
			measurementsList.push(["leftEyeHeight",	"bbox", getLayerParam("Adobe.Face.LeftPupilRange"), "Y"]);
			measurementsList.push(["leftEyeHeight",	"bbox", getHandleParentLayerFallback("Adobe.Face.LeftEye"), "Y"]);
			
			measurementsList.push(["rightEyeHeight",	"bbox", getLayerParam("Adobe.Face.RightPupilRange"), "Y"]);
			measurementsList.push(["rightEyeHeight",	"bbox", getHandleParentLayerFallback("Adobe.Face.RightEye"), "Y"]);		

			measurementsList.push(["leftPupilWidth",	"bbox", getLayerParam("Adobe.Face.LeftPupilSize"), "X"]);
			measurementsList.push(["leftPupilHeight", "bbox", getLayerParam("Adobe.Face.LeftPupilSize"), "Y"]);

			measurementsList.push(["rightPupilWidth", "bbox", getLayerParam("Adobe.Face.RightPupilSize"), "X"]);
			measurementsList.push(["rightPupilHeight", "bbox", getLayerParam("Adobe.Face.RightPupilSize"), "Y"]);		
		}

		if (includeFaceB) {
			measurementsList.push(["leftEyeEyebrowDist",	"pos", getHandleParam("Adobe.Face.LeftEye"),	getHandleParam("Adobe.Face.LeftEyebrow"),		"Y"]);
			measurementsList.push(["leftEyeEyebrowDist",	"scaledCopy", 0.15, "headHeight"]); // fallback if both "Left Eye" and "Left Eyebrow" handles don't exist
																			// maybe remove this -- is it useful to have eyebrows without eyes?

			measurementsList.push(["rightEyeEyebrowDist", "pos", getHandleParam("Adobe.Face.RightEye"),		getHandleParam("Adobe.Face.RightEyebrow"),	"Y"]);
			measurementsList.push(["rightEyeEyebrowDist", "scaledCopy", 0.15, "headHeight"]); // fallback if both "Right Eye" and "Right Eyebrow" handles don't exist
			
			measurementsList.push(["interocularDist",		"pos", getHandleParam("Adobe.Face.LeftEye"),		getHandleParam("Adobe.Face.RightEye"),		"X"]);
			measurementsList.push(["interocularDist",		"scaledCopy", 0.25, "headWidth"]);
			
			measurementsList.push(["noseDepth",		"scaledCopy", 0.15, "interocularDist"]);
			measurementsList.push(["eyeDepth",		"scaledCopy", 0.09, "interocularDist"]);
			
			// controls distance that the left eyelid top moves down, and bottom moves up; primary: space between top & bottom bounds
			measurementsList.push(["leftEyelidDist",	"bboxDiff", getLayerParam("Adobe.Face.LeftEyelidTopLayer"), getLayerParam("Adobe.Face.LeftEyelidBottomLayer"), "Y"]);
			//	fallback: distance between top & bottom handles
			measurementsList.push(["leftEyelidDist",	"pos", getHandleParam("Adobe.Face.LeftEyelidTop"), getHandleParam("Adobe.Face.LeftEyelidBottom"), "Y"]);
			
			measurementsList.push(["rightEyelidDist", "bboxDiff", getLayerParam("Adobe.Face.RightEyelidTopLayer"), getLayerParam("Adobe.Face.RightEyelidBottomLayer"), "Y"]);
			measurementsList.push(["rightEyelidDist", "pos", getHandleParam("Adobe.Face.RightEyelidTop"), getHandleParam("Adobe.Face.RightEyelidBottom"), "Y"]);
		}
		
		if (includeMouthB) {
			// $$$ TODO: the 0 below implies there is always a Neutral mouth in the first Mouth group
			// note: the sMAP parent for mouth is not view, only Mouth group
			measurementsList.push(["mouthWidth", "bbox", getLayer(args, "Adobe.Face.NeutralMouth", 0), "X"]); // TODO: change to mouth group?			
			measurementsList.push(["mouthHeight", "bbox", getLayer(args, "Adobe.Face.NeutralMouth", 0), "Y"]);			
		}

		for (i = 0; i < measurementsList.length; i += 1) {
			m = measurementsList[i];

			if (!(puppetMeasurements.hasOwnProperty(m[0]) && puppetMeasurements[m[0]])) {
				value = null;

				if (m[1] === "pos" && m[2] && m[3] && m[4]) {
					// rootMatrix is a 3x3 transformation (in homogeneous coords)
					// rootMatrix[6] = x translation, rootMatrix[7] = y translation 
					if (m[4] === "X") { 
						j = 6; 
					}
					else { 
						j = 7; 
					}
					// TODO: may not work as intended when Puppets are scaled: ten times smaller or ten times larger
					value = Math.abs(args.getHandleMatrixRelativeToScene(m[2])[j]-args.getHandleMatrixRelativeToScene(m[3])[j]);
				} 
				else if (m[1] === "bbox" && m[2] && m[3]) {
					bbox1 = getBBox(m[2]);

					if (m[3] === "X") { 
						value = bbox1[2]; 
					}
					else { 
						value = bbox1[3]; 
					}
				}
				else if (m[1] === "bboxDiff" && m[2] && m[3] && m[4]) {

					bbox1 = getBBox(m[2]);
					bbox2 = getBBox(m[3]);

					if (m[4] === "X") {
						value = bbox2[0] - (bbox1[0] + bbox1[2]);
					}
					else {
						value = bbox2[1] - (bbox1[1] + bbox1[3]);
					}
					if (value <= 0) {
						value = null;	// will fall back on "pos" instead
					}
				}
				else if (m[1] === "scaledCopy" && m[2] && m[3]) {
					value = m[2] * puppetMeasurements[m[3]];
				}

				puppetMeasurements[m[0]] = value;

				// DEBUG
				//console.logToUser(m[0] + "_" + m[1] + " = " + puppetMeasurements[m[0]]);
			}
		}
		
		utils.assert(self.aPuppetMeasurements.length === viewIndex);
		self.aPuppetMeasurements.push(puppetMeasurements);

		// add additional compound measurements; note: reads aPuppetMeasurements
		computeEyeGazeRanges(self, args, viewIndex);
	}
	
	function SmoothingRingBuffer () {
		this.ringBufferSize = 3;
		this.ringBufferA = [null, null, null];	// To reduce latency, we always use the most recent values as our 4 variable.
		this.insertionIndex = 0;
		this.lastInsertionTime = null;
	}

	utils.mixin(SmoothingRingBuffer, {
		getInsertionIndex : function () {
			return this.insertionIndex;
		},
		getLastInsertionTime : function () {
			return this.lastInsertionTime;
		},
		setLastInsertionTime : function (t) {
			this.lastInsertionTime = t;
		},
		incrementInsertionIndex : function () {
			this.insertionIndex = (this.getInsertionIndex()+1) % this.ringBufferSize;
		},
		addCurrentSample : function (currentValueArray, sampleTime, sampleInterval) {
			var insertKeyframeB = false, keyFramesA = [], 
					dt, interpTime;

			insertKeyframeB = (this.getLastInsertionTime() === null);
			if (!insertKeyframeB) {
				dt = sampleTime - this.getLastInsertionTime();

				insertKeyframeB = (dt >= sampleInterval);
			}
			if (insertKeyframeB) {
				dt = 0;
				this.ringBufferA[this.getInsertionIndex()] = currentValueArray;
				this.incrementInsertionIndex();
				this.setLastInsertionTime(sampleTime);
			}

			keyFramesA[0] = currentValueArray;
			for (var i = 1; i < this.ringBufferSize+1; i++) {
				var idx = (this.getInsertionIndex() - i + this.ringBufferSize);
				idx = idx % this.ringBufferSize;
				keyFramesA[i] = this.ringBufferA[idx];

				if (keyFramesA[i] === null) {
					utils.assert(i > 0);
					keyFramesA[i] = keyFramesA[i-1];
				}
			}

			interpTime = 1 - (dt / sampleInterval);
			interpTime = Math.max(0.0, interpTime);
			interpTime = Math.min(1.0, interpTime);

			var smoothValuesArray = [];
			for (var label in currentValueArray) {
				if (currentValueArray.hasOwnProperty(label)) {
					
					if (canIterpolateHeadLabel(label)) {
						var x0 = keyFramesA[0][label], x1 = keyFramesA[1][label], 
							x2 = keyFramesA[2][label], x3 = keyFramesA[3][label];

						smoothValuesArray[label] = mathUtils.cubicLerp(x0, x1, x2, x3, interpTime, 0.75);
					} else {
						smoothValuesArray[label] = keyFramesA[0][label];
					}
				}
			}
			return smoothValuesArray;
		}
	});
	
	function SmoothingFilter () {
		this.smoothingRingBuffer = new SmoothingRingBuffer();
		this.currentPose0 = null;
		this.previousPose0 = null;
		this.lastValue0 = null;
		this.lastPauseKeyState = 0;

		// Used to store data related to transitioning between current value and pose, then pose back to current. 
		this.transitionStartTime = 0;
		this.transitionEndTime = 0;
	}

	utils.mixin(SmoothingFilter, {

		getData : function (inCurrentTime, inSmoothingRate, inCurrentValuesData, inPauseKeyDown) {
			var smoothingRate = inSmoothingRate, smoothingRateHz, filterValues, currentValues = inCurrentValuesData,
				pauseKeyDown = inPauseKeyDown, currentTime = inCurrentTime, normalizedHz;

			// 100% means 1Hz; < 1% means off; 1% is 30Hz.
			if (smoothingRate < 1) {
				smoothingRate = 0.0;
			}
			normalizedHz = ((smoothingRate - 1) / 99);

			// This helps linearize the data:
			normalizedHz = Math.sqrt(Math.sqrt(normalizedHz));
			smoothingRateHz = smoothingRate ? ((1 - normalizedHz) * 29 + 1) : 0.0;
//			console.logToUser("smoothingRateHz = " + smoothingRateHz);

			if (this.lastPauseKeyState !== pauseKeyDown) {
				var fadeDuration = (smoothingRateHz ? 3 / smoothingRateHz : 0.25),  // Make this a parameter?
					resetTransitionTimesB = true;

				// Pause state changed.
				if (pauseKeyDown) {
					var	inTransitionB = currentTime > this.transitionStartTime && currentTime < this.transitionEndTime;

					if (!inTransitionB) {
						fadeDuration = (smoothingRateHz ? 1 / smoothingRateHz : 0.0);	
						this.currentPose0 = currentValues;
					} else {
						this.currentPose0 = currentValues;
						resetTransitionTimesB = false;
					}
					this.previousPose0 = this.lastValue0; 
				} else {
					this.currentPose0 = this.lastValue0;
				}

				if (resetTransitionTimesB) {
					this.transitionStartTime = currentTime;
					this.transitionEndTime = currentTime + fadeDuration;
				}
			}
			this.lastPauseKeyState = pauseKeyDown;

			if (smoothingRateHz) {
				filterValues = this.smoothingRingBuffer.addCurrentSample(currentValues, currentTime, 1 / smoothingRateHz);
			} else {
				filterValues = currentValues;
			}

			if (this.currentPose0 && this.previousPose0) {
				var	interpB = currentTime >= this.transitionStartTime && currentTime < this.transitionEndTime, 
					startKey = pauseKeyDown ? this.previousPose0 : this.currentPose0, endKey = pauseKeyDown ? this.currentPose0 : filterValues;

				if (interpB) {
					var interpTime = (currentTime - this.transitionStartTime) / (this.transitionEndTime - this.transitionStartTime);

					interpTime = mathUtils.easeInAndOut(interpTime);

					var interpValues = [];
					for (var label in startKey) {
						if (startKey.hasOwnProperty(label)) {
							if (canIterpolateHeadLabel(label)) {
								interpValues[label] = mathUtils.lerp(startKey[label], endKey[label], interpTime);		
							} else {
								interpValues[label] = endKey[label];
							}
						}
					}

					filterValues = interpValues;
				} else {
					filterValues = (currentTime === this.transitionStartTime) ? startKey : endKey;
				}
			}

			this.lastValue0 = filterValues;
			
			return filterValues;
		},
	});	
	
	// simple ring buffer
	function RingBuffer (inBufferSize) {
		this.ringBufferSize = inBufferSize;
		this.ringBufferA = [];
		this.insertionIndex = 0;
		this.lastInsertionTime = null;

		for (var i = 0; i < this.ringBufferSize; i++) {
			this.ringBufferA.push(null);
		}
	}

	utils.mixin(RingBuffer, {
		getData : function () {
			var dataA = [];
			for (var i = 0; i < this.ringBufferSize; i++) {
				var idx = (this.getInsertionIndex() + i) % this.ringBufferSize;
				if (this.ringBufferA[idx]) {
					dataA.push(this.ringBufferA[idx]);
				}
			}
			return dataA;
		},
		getInsertionIndex : function () {
			return this.insertionIndex;
		},
		getLastInsertionTime : function () {
			return this.lastInsertionTime;
		},
		setLastInsertionTime : function (t) {
			this.lastInsertionTime = t;
		},
		incrementInsertionIndex : function () {
			this.insertionIndex = (this.getInsertionIndex()+1) % this.ringBufferSize;
		},
		addCurrentSample : function (currentValueArray, sampleTime, sampleInterval) {
			var insertKeyframeB = false, dt;

			insertKeyframeB = (this.getLastInsertionTime() === null);
			if (!insertKeyframeB) {
				dt = sampleTime - this.getLastInsertionTime();

				insertKeyframeB = (dt >= sampleInterval);
			}
			if (insertKeyframeB) {
				dt = 0;
				this.ringBufferA[this.getInsertionIndex()] = currentValueArray;
				this.incrementInsertionIndex();
				this.setLastInsertionTime(sampleTime);
			}
		}
	});

	// pose-to-pose filter, operating on a specified set of head14 labels with given thresholds

	// default constants for pose to pose filter expressed as ranges where first entry 
	// corresponds to filter level = 0% and last entry to filter level = 100%

	var kPoseToPoseFilterDurationRangeA = [0, 0.7];						// duration (in seconds)
	//
	var kPoseToPoseFilterMinIntervalRangeA = [0, 1.0];					// thresholds on min/max time intervals allowed between pose transitions
	var kPoseToPoseFilterMaxIntervalRangeA = [3.0, 8.0];
	// 
	var kPoseToPoseFilterHeadPosMeanThreshRangeA = [0, 0.3];			// thresholds on means/variances for various pose dimensions
	var kPoseToPoseFilterHeadRotMeanThreshRangeA = [0, 0.3];	
	var kPoseToPoseFilterEyebrowPosMeanThreshRangeA = [0, 0.1];
	// 	
	var kPoseToPoseFilterHeadPosVarThreshRangeA = [0.01, 0.01];			// NOTE: modifying variance thresholds don't seem to have a big impact, so we keep them constant for now ...
	var kPoseToPoseFilterHeadRotVarThreshRangeA = [0.01, 0.01];
	var kPoseToPoseFilterEyebrowPosVarThreshRangeA = [0.05, 0.05];

	function PoseToPoseFilter () {
		this.poseToPoseLabels = [
			"Head/DX", 
			"Head/DY", 
			"Head/DZ", 
			"Head/Orient/X", 
			"Head/Orient/Y", 
			"Head/Orient/Z", 
			"Head/LeftEyebrow",
			"Head/RightEyebrow",
			"Head/Scale"
		];

		this.sceneFrameRate = 0;
		this.filterDuration = 0;			
		this.filterSize = 0;
		this.minInterval = null;
		this.maxInterval = null;
		this.lastPoseChangeTime = 0;
		this.ringBuffer = null;
		this.filteredLabelsA = null;
		this.poseInfoA = null;

		this.filterLevel = 0;				// cache of user-facing filter level param (in %): 0 means no filtering
		this.minPoseDuration = 0;			// cache of user-facing min pose duration (in s) param

		this.advancedParams = {				// cache of advanced params (matching hidden param ids in Face behavior)
			filterNumSamples: 0,
			minInterval: 0,
			maxInterval: 0, 
			headRotMeanThresh: 0,
			headRotVarThresh: 0,
			headPosMeanThresh: 0, 
			headPosVarThresh: 0,
			eyebrowPosMeanThresh: 0,
			eyebrowPosVarThresh: 0
		};
	}

	utils.mixin(PoseToPoseFilter, {

		setLastPoseChangeTime : function(inTime) {
			this.lastPoseChangeTime = inTime;
		},

		setThresholds : function(inHead14Thresholds) {
			var thresholds = inHead14Thresholds;

			// map stores mean, variance and thresholds for pose transitions per entry
			this.filteredLabelsA = [];
			this.poseInfoA = []; 
			for (var label in thresholds) {
				if (thresholds.hasOwnProperty(label)) {

					this.poseInfoA.push({
						label: label,
						currentMean: null,
						currentVariance: 0, 
						meanThresh: thresholds[label].meanThresh,
						varianceThresh: thresholds[label].varianceThresh
					});

					this.filteredLabelsA.push(label);
				}
			}

			//console.log("poseInfoMap: " + JSON.stringify(this.poseInfoA));
		},

		updateThresholds : function(inThresholds) {
			var	poseToPoseLabels = this.poseToPoseLabels, 
				poseToPoseThresholds = {}, label, labelMeanThresh, labelVarThresh, 
				posMeanThresh = inThresholds.headPosMeanThresh, 
				posVarThresh = inThresholds.headPosVarThresh,
				rotMeanThresh = inThresholds.headRotMeanThresh, 
				rotVarThresh = inThresholds.headRotVarThresh,
				eyebrowPosMeanThresh = inThresholds.eyebrowPosMeanThresh, 
				eyebrowPosVarThresh = inThresholds.eyebrowPosVarThresh;

			for (var i = 0; i < poseToPoseLabels.length; ++i) {
				label = poseToPoseLabels[i];

				// get appropriate thresholds from behavior params
				if ((label.indexOf("Head/D") >= 0) || (label === "Head/Scale")) {
					labelMeanThresh = posMeanThresh;
					labelVarThresh = posVarThresh;
				}
				else if (label.indexOf("Head/Orient/") >= 0) {
					labelMeanThresh = rotMeanThresh;
					labelVarThresh = rotVarThresh;
				}
				else if (label.indexOf("Eyebrow") >= 0) {
					labelMeanThresh = eyebrowPosMeanThresh;
					labelVarThresh = eyebrowPosVarThresh;
				}
				else {
					// invalid/unexpected threshold label
					utils.assert(false);
				}

				poseToPoseThresholds[poseToPoseLabels[i]] = {
					meanThresh: labelMeanThresh,
					varianceThresh: labelVarThresh
				};
			}
			
			this.setThresholds(poseToPoseThresholds);
		},

		updateIntervals : function(inIntervals) {
			this.minInterval = inIntervals.minInterval;
			this.maxInterval = inIntervals.maxInterval;
		},

		updateAdvancedParams : function(inAdvancedParams) {
			var paramsChanged = false, paramName, paramValue, inParamValue;

			for (paramName in this.advancedParams) {
				if (this.advancedParams.hasOwnProperty(paramName)) {
					paramValue = this.advancedParams[paramName];
					inParamValue = inAdvancedParams[paramName];
					if (paramValue !== inParamValue) {
						paramsChanged = true;
						this.advancedParams[paramName] = inParamValue;

						// if num filter samples changed, rebuild ring buffer of appropriate size
						if (paramName === "filterNumSamples") {
							utils.assert(this.sceneFrameRate > 0);

							this.filterSize = Math.ceil(inParamValue);
							this.filterDuration = this.filterSize / this.sceneFrameRate;
							this.ringBuffer = new RingBuffer(this.filterSize);
						}

						// handle intervals separately from other thresholds
						if (paramName === "minInterval") {
							this.minInterval = inParamValue;
						}
						if (paramName === "maxInterval") {
							this.maxInterval = inParamValue;
						}						
					}
				}
			}

			if (paramsChanged) {
				this.updateThresholds(inAdvancedParams);
			}

			return paramsChanged;
		},

		updateUserFacingParams : function(inSceneFrameRate, inFilterLevel, inMinPoseDuration) {
			var frameRateChangedB = (inSceneFrameRate !== this.sceneFrameRate),
				filterLevelChangedB = (inFilterLevel !== this.filterLevel),
				minPoseDurationChangedB = (inMinPoseDuration !== this.minPoseDuration);

			// if frame rate or filter level have changed, update filter duration and size
			if (frameRateChangedB || filterLevelChangedB) {

				this.filterDuration = this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterDurationRangeA);
				var newFilterSize = Math.ceil(this.filterDuration * inSceneFrameRate);

				// if filter size has changed, rebuild ring buffer of appropriate size
				if (this.filterSize !== newFilterSize) {
					this.filterSize = newFilterSize;
					this.ringBuffer = new RingBuffer(this.filterSize);
				}
			}

			// if filter level has changed, update filter thresholds
			if (filterLevelChangedB) {
				// now exposing min interval as explicit, user-facing param
				//var newIntervals = this.getIntervalsFromFilterLevel(inFilterLevel);
				var newThresholds = this.getThresholdsFromFilterLevel(inFilterLevel);
				this.updateThresholds(newThresholds);
			}

			// if min pose duration has changed, update intervals
			if (minPoseDurationChangedB) {
				var newIntervals = { minInterval: inMinPoseDuration, maxInterval: null };
				this.updateIntervals(newIntervals);
			}

			// update cached frame rate, min pose duration, and filter level
			this.sceneFrameRate = inSceneFrameRate;
			this.filterLevel = inFilterLevel;
			this.minPoseDuration = inMinPoseDuration;

			return (frameRateChangedB || filterLevelChangedB);
		},

		getConstantFromFilterLevel : function (inFilterLevel, inConstantRangeA) {
			var filterFactor = inFilterLevel / 100;
			return inConstantRangeA[0] + (filterFactor * (inConstantRangeA[1] - inConstantRangeA[0]));
		},

		getIntervalsFromFilterLevel : function (inFilterLevel) {
			return {
				minInterval: this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterMinIntervalRangeA),
				maxInterval: this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterMaxIntervalRangeA)
			};
		},

		getThresholdsFromFilterLevel : function (inFilterLevel) {
			return {
				headPosMeanThresh: 		this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterHeadPosMeanThreshRangeA),
				headRotMeanThresh: 		this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterHeadRotMeanThreshRangeA), 
				eyebrowPosMeanThresh: 	this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterEyebrowPosMeanThreshRangeA),
				headPosVarThresh: 		this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterHeadPosVarThreshRangeA),
				headRotVarThresh: 		this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterHeadRotVarThreshRangeA), 
				eyebrowPosVarThresh: 	this.getConstantFromFilterLevel(inFilterLevel, kPoseToPoseFilterEyebrowPosVarThreshRangeA)
			};
		},

		getMeans : function () {
			var meansA = [], data = this.ringBuffer.getData();
			
			if (data.length > 0) {	
				var i, j, sumsA = data[0].slice(0);
				for (i = 1; i < data.length; i++) {
					var valueA = data[i];
					for (j = 0; j < valueA.length; j++) {
						sumsA[j] += valueA[j];
					}
				}

				meansA = sumsA;
				for (j = 0; j < sumsA.length; j++) {
					meansA[j] = sumsA[j] / data.length;
				}

			}

			return meansA;
		},

		getVariances : function () {
			var variancesA = [], data = this.ringBuffer.getData(), meansA = this.getMeans();

			if (data.length > 0) {
				var i, j, sumsA = [], firstValueA = data[0];
				for (j = 0; j < firstValueA.length; j++) {
					sumsA[j] = 0;
					variancesA[j] = 0;
				}

				if (data.length > 1) {
					for (i = 0; i < data.length; i++) {
						var valueA = data[i];
						for (j = 0; j < valueA.length; j++) {
							sumsA[j] += Math.pow(valueA[j] - meansA[j], 2);
						}
					}	

					variancesA = sumsA;

					for (j = 0; j < sumsA.length; j++) {
						variancesA[j] = sumsA[j] / (data.length - 1);
					}
				}
			}

			return variancesA;
		},

		getData : function (inCurrentTime, inSmoothingRate, inCurrentValuesData) {
			var currentValues = inCurrentValuesData, filterValues = {}, valuesToFilterA,
				currentTime = inCurrentTime, channelInfo, meanDifference, i;

			// select out relevant values
			valuesToFilterA = this.poseInfoA.reduce(function(selectedValuesA, value) {
				return selectedValuesA.concat(currentValues[value.label]);
			}, []);

			// add samples
			utils.assert(this.sceneFrameRate > 0);
			this.ringBuffer.addCurrentSample(valuesToFilterA, currentTime, 1 / (2 * this.sceneFrameRate));

			var meansA = this.getMeans();
			var variancesA = this.getVariances();

			utils.assert(meansA.length === variancesA.length);
			utils.assert(variancesA.length === this.filteredLabelsA.length); 
			utils.assert(this.filteredLabelsA.length === this.poseInfoA.length);

			// update pose if:
			// - we've waited longer than this.maxInterval 
			// OR
			// - we've waited longer than this.minInterval AND
			// - at least one mean is above the threshold, AND
			// - all variances are below the threshold

			var dt = (currentTime - this.lastPoseChangeTime), 
				updatePoseB = false, meansChangedB = false, variancesLowB = true,
				longerThanMinIntervalB = (this.minInterval === null) || (dt > this.minInterval),
				longerThanMaxIntervalB = (this.maxInterval !== null) && (dt > this.maxInterval);

			// if we've waited longer than maxInterval, update pose without checking anything else
			if (longerThanMaxIntervalB) {
				updatePoseB = true;
			}
			// if we've waited longer than minInterval, check means and variances against thresholds
			else if (longerThanMinIntervalB) {
				for (i = 0; i < this.poseInfoA.length; i++) {
					channelInfo = this.poseInfoA[i];	

					// if we find any invalid, uninitialized channels, then update the whole pose
					if (channelInfo.currentMean === null) {
						updatePoseB = true;
						break;
					}

					meanDifference = Math.abs(channelInfo.currentMean - meansA[i]);
					if (meanDifference > channelInfo.meanThresh) {
						meansChangedB = true;
					}

					if (variancesA[i] > channelInfo.varianceThresh) {
						variancesLowB = false;
					}
				}

				updatePoseB = (updatePoseB || (meansChangedB && variancesLowB));
			}

			if (updatePoseB) {
				this.setLastPoseChangeTime(currentTime);
				for (i = 0; i < this.poseInfoA.length; i++) {
					channelInfo = this.poseInfoA[i];
					channelInfo.currentMean = meansA[i];
					channelInfo.currentVariance = variancesA[i];
				}
			}

/*
			// DEBUG: output current values
			var logStr = "," + JSON.stringify(inCurrentTime);
			for (i = 0; i < valuesToFilterA.length; ++i) {
				logStr += ",";
				logStr += JSON.stringify(valuesToFilterA[i]);
			}

			// DEBUG: and means, variances
			for (i = 0; i < meansA.length; ++i) {
				logStr += ",";
				logStr += JSON.stringify(meansA[i]);
			}
			for (i = 0; i < variancesA.length; ++i) {
				logStr += ",";
				logStr += JSON.stringify(variancesA[i]);
			}

			console.log(logStr);
*/

			// update relevant values
			var filterValueIndex;
			for (var label in currentValues) {
				if (currentValues.hasOwnProperty(label)) {
					filterValueIndex = this.filteredLabelsA.indexOf(label);
					if (filterValueIndex >= 0 && (this.poseInfoA[filterValueIndex].currentMean !== null)) {
						filterValues[label] = this.poseInfoA[filterValueIndex].currentMean;						
					} else {
						filterValues[label] = currentValues[label];
					}
				}
			}

			return filterValues;
		},
	});	

	// compute eye dart directions as points around the perimeter of an ellipse
	// centered at the origin, with major x-axis dimension a and minor y-	axis dimension b
	function getEyeDartDirectionFromAngle(angleRad) {
		var x, y, tanAngle, epsilon = 0.00001, a = 1.0, b = 0.8;

		if (Math.abs(angleRad) < epsilon) {
			x = a;
			y = 0;
		}
		else if (Math.abs(angleRad - (0.5 * Math.PI)) < epsilon) {
			x = 0;
			y = b;
		}
		else if (Math.abs(angleRad - Math.PI) < epsilon) {
			x = -a;
			y = 0;
		}
		else if (Math.abs(angleRad - (1.5 * Math.PI)) < epsilon) {
			x = 0;
			y = -b;
		}

		else {
			tanAngle = Math.tan(angleRad);
			x = (a*b) / (Math.sqrt(b*b + a*a*tanAngle*tanAngle));

			if (angleRad > (0.5 * Math.PI) && angleRad < (1.5 * Math.PI)) {
				x = -x;
			}

			y = x * tanAngle;
		}

		return [x, y];
	}

	function getEyeDartDirections(numEyeDartDirections) {
		var angleRad, direction, directions = [];

		for (var i=0; i<numEyeDartDirections; ++i) {
			angleRad = (i/numEyeDartDirections) * 2.0 * Math.PI;
			direction = getEyeDartDirectionFromAngle(angleRad);
			directions.push(direction);
		}		

		return directions;
	}

	function getEyeDartDirectionScaleFactors(eyeDartDirections, leftRightScaleFactor, upDownScaleFactor, offAxisScaleFactor) {
		var directionScaleFactors = [], numEyeDartDirections = eyeDartDirections.length, direction, epsilon = 0.00001;

		for (var i=0; i<numEyeDartDirections; ++i) {
			direction = eyeDartDirections[i];

			// left-right
			if (Math.abs(direction[0]) > epsilon && Math.abs(direction[1]) < epsilon) {
				directionScaleFactors.push(leftRightScaleFactor);
			}
			// up-down
			else if (Math.abs(direction[0]) < epsilon && Math.abs(direction[1]) > epsilon) {
				directionScaleFactors.push(upDownScaleFactor);
			}
			// off-axis
			else {
				directionScaleFactors.push(offAxisScaleFactor);
			}
		}

		return directionScaleFactors;
	}

	function snapToPredefinedEyeDartDirection(self, currentTime, eyeDartParams, offset) {
		var numEyeDartDirections = eyeDartParams.numEyeDartDirections,
			minSnapOffsetDist = eyeDartParams.minSnapOffsetDist,
			minGazeDuration = eyeDartParams.minGazeDuration,
			leftRightScaleFactor = eyeDartParams.leftRightScaleFactor,
			upDownScaleFactor = eyeDartParams.upDownScaleFactor,
			offAxisScaleFactor = eyeDartParams.offAxisScaleFactor,
			directions, direction, distFromCenter, distFromLastOffset, directionScaleFactors,
			gazeTooShortB, distToDirection, minDistToDirection = null, 
			snappedOffset = offset, epsilon = 0.00001;

		// if no eye dart directions specified, don't snap
		if (numEyeDartDirections === 0) {
			return offset;
		}

		// if minimum gaze duration has not elapsed, preserve last gaze direction
		gazeTooShortB = ((self.lastEyeDartTime !== null) && ((currentTime - self.lastEyeDartTime) < minGazeDuration));
		if (gazeTooShortB) {
			return self.lastEyeDartOffset;
		}

		// if offset is too close to center, snap to center
		distFromCenter = v2.magnitude(offset);
		if (distFromCenter < minSnapOffsetDist) {
			v2.initWithEntries(0.0, 0.0, snappedOffset);
		}

		// otherwise, snap to closest predefined eye dart direction
		else {
			// NOTE: we could hardcode the eye directions or initialize once, 
			// but this is a pretty lightweight computation and 
			// recomputing gives us the flexibility to support user
			// customization of the granularity of eye darts
			directions = getEyeDartDirections(numEyeDartDirections);
			directionScaleFactors = getEyeDartDirectionScaleFactors(directions, leftRightScaleFactor, upDownScaleFactor, offAxisScaleFactor);

			// pick nearest predefined eye dart direction based on offset
			for (var i=0; i<numEyeDartDirections; ++i) {
				direction = directions[i];
				distToDirection = v2.distance(direction, offset);
				distToDirection *= directionScaleFactors[i];
				if (minDistToDirection === null || distToDirection < minDistToDirection) {
					snappedOffset = direction;
					minDistToDirection = distToDirection;	
				}
			}
		}

		// update last eye dart
		distFromLastOffset = v2.distance(self.lastEyeDartOffset, snappedOffset);
		if (distFromLastOffset > epsilon) {
			self.lastEyeDartTime = currentTime;
			self.lastEyeDartOffset = v2.clone(snappedOffset);
		}

		return v2.clone(snappedOffset);
	}

	// snap eye gaze in head14 to predefined eye dart directions:
	// numEyeDartDirections is the number of evenly spaced snapping directions around centre of each eye
	// minSnapOffsetDist is the minimum distance away from centre required to snap to one of the dart directions (normalized to the radius of the eye)
	function snapEyeGaze(self, currentTime, eyeDartParams, head14) {
		var outputHead14 = head14, leftEyeGazeOffset, rightEyeGazeOffset, snappedLeftEyeGazeOffset, snappedRightEyeGazeOffset;

		leftEyeGazeOffset = v2.initWithEntries(head14["Head/LeftEyeGazeX"], head14["Head/LeftEyeGazeY"]);
		rightEyeGazeOffset = v2.initWithEntries(head14["Head/RightEyeGazeX"], head14["Head/RightEyeGazeY"]);		

		snappedLeftEyeGazeOffset = snapToPredefinedEyeDartDirection(self, currentTime, eyeDartParams, leftEyeGazeOffset);
		snappedRightEyeGazeOffset = snapToPredefinedEyeDartDirection(self, currentTime, eyeDartParams, rightEyeGazeOffset);		

		outputHead14["Head/LeftEyeGazeX"] = snappedLeftEyeGazeOffset[0];
		outputHead14["Head/LeftEyeGazeY"] = snappedLeftEyeGazeOffset[1];
		//		
		outputHead14["Head/RightEyeGazeX"] = snappedRightEyeGazeOffset[0];
		outputHead14["Head/RightEyeGazeY"] = snappedRightEyeGazeOffset[1];		

		return outputHead14;
	}

	function getHead14(args) {
		var label, v, vals = {}, inputId = "cameraInput";

		// use first label existence as a proxy for all existing
		if (args.getParamEventValue(inputId, "Head/InputEnabled", null, false)) {
			args.setEventGraphParamRecordingValid(inputId);
			for (label in head14Labels) {
				if (head14Labels.hasOwnProperty(label)) {
				    v = args.getParamEventValue(inputId, label, null, head14Labels[label]);
					if (v !== undefined) {
						vals[label] = v;
					} else { // else  missing part of head14 data 
						vals[label] = head14Labels[label];
					}
				}
			}
		} else {
			// return identity head14 -- perhaps camera tracking not spun up yet
			for (label in head14Labels) {
				if (head14Labels.hasOwnProperty(label)) {
					vals[label] = head14Labels[label];
				}
			}
		}
		
		return vals;
	}

	function getFilteredHead14(args) {
		var label, v, vals = {}, inputId = "cameraInput", k = args.getParamEventOutputKey(inputId), labelKey;

		// use first label existence as a proxy for all existing
		for (label in head14Labels) {
			if (head14Labels.hasOwnProperty(label)) {
				labelKey = k + "Filtered/Head/" + label;
				if (canIterpolateHeadLabel(label)) {
					v = args.getParamEventValue(inputId, labelKey, null, head14Labels[label]);
				} else {
					var graphEvaluatorArray = args.getGraphEvaluatorArray(inputId);
					if (graphEvaluatorArray.length) {
						v = args.getParamEventValueFromEvaluator(graphEvaluatorArray[0], inputId, labelKey);
					} else {
						v = undefined;
					}
				}
				
				if (v !== undefined) {
					vals[label] = v;
				} else { // else  missing part of head14 data 
					vals[label] = head14Labels[label];
				}
			}
		}
		
		return vals;
	}
	
	function computeInitTransforms (self, args, view, viewIndex) {
		var faceFeatureLabel, handleParam,
			featureLabels = self.aFeatureLabels[viewIndex],
			puppetInitTransforms = {};

		for (faceFeatureLabel in featureLabels) {
			if (featureLabels.hasOwnProperty(faceFeatureLabel)) {
				handleParam = getHandle(args, faceFeatureLabel, viewIndex);
				if (handleParam) {
					// WL: we are explicitly recording whether a face feature label 
					// is present ... as an alternative, we could also just check 
					// if puppetInitTransforms has an entry for the label.
					featureLabels[faceFeatureLabel] = true;

					puppetInitTransforms[faceFeatureLabel] = {
						"position": [0, 0],
						"scale"	: [1, 1],
						"angle"	: 0
					};
				}
			}
		}
		
		utils.assert(self.aPuppetInitTransforms.length === viewIndex); // assumes we're called in view order
		self.aPuppetInitTransforms.push(puppetInitTransforms);
	}
	
	function applyPuppetTransforms(self, args, viewIndex, inTransforms) {
		var faceFeatureLabel, transform, initState, faceFeatureNode, 
			dof = [], position = [], scale = [], shear = [0], angle;

		for (faceFeatureLabel in inTransforms) {
			if (inTransforms.hasOwnProperty(faceFeatureLabel)) {
				transform = inTransforms[faceFeatureLabel];
				initState = self.aPuppetInitTransforms[viewIndex][faceFeatureLabel];

				faceFeatureNode = getHandle(args, faceFeatureLabel, viewIndex);

				if (transform && initState && faceFeatureNode) {
					v2.add(initState.position, v2.scale(1, transform.translate), position);
					// TODO: will initState ever have a scale != 1 that we have to take into account?
					scale = transform.scale;
					angle = initState.angle + transform.angle;

					var transformTask = {
						x : position[0],
						y : position[1],
						xScale : scale[0],
						yScale : scale[1],
						angle : angle
					};

					// FIXME: connect with blend weight...
					var	weight = 1.0,
						move = new tasks.MoveTo(transformTask);
					tasks.handle.attachTask(faceFeatureNode, move, weight);
				}
			}
		}
	}

	// Clear the rehearsal state
	function onResetRehearsalData (self) {
		self.faceSmoothingFilter = new SmoothingFilter();
		self.facePoseToPoseFilter = null;
	}

	function updateFacePoseToPoseFilterParams(self, args) {

		// update user-facing params
		self.facePoseToPoseFilter.updateUserFacingParams(args.scene_.getFrameRate(), args.getParam("poseFilteringLevel"), args.getParam("minPoseDuration"));

		// if using manual override controls, check if those hidden params have changed
		if (args.getParam("enablePoseToPoseAdvancedParams")) {
			var advancedParams = {
				filterNumSamples: 		args.getParam("filterNumSamples"),
				minInterval: 			args.getParam("minInterval"),
				maxInterval: 			args.getParam("maxInterval"), 
				headRotMeanThresh: 		args.getParam("headRotMeanThresh"),
				headRotVarThresh: 		args.getParam("headRotVarThresh"),
				headPosMeanThresh: 		args.getParam("headPosMeanThresh"), 
				headPosVarThresh: 		args.getParam("headPosVarThresh"),
				eyebrowPosMeanThresh: 	args.getParam("eyebrowPosMeanThresh"),
				eyebrowPosVarThresh: 	args.getParam("eyebrowPosVarThresh")
			};

			self.facePoseToPoseFilter.updateAdvancedParams(advancedParams);
		}
	}

	function onCreateStageBehavior (self, args, includeFaceB, includePupilsB, includeMouthB) {
		onResetRehearsalData(self);

		var headHandle, newHead = null;
		args.getParam = args.getStaticParam;	// @@@HACK! until we rename this function

		// these arrays are all indexed by view
		self.aPuppetInitTransforms = [];
		self.aPuppetMeasurements = [];
		self.aFeatureLabels = [];

		headHandle = args.getParam(makeHandleIdFromLabel("Adobe.Face.Head"))[0];
		if (headHandle) {
			newHead = headHandle.getPuppet(); // get puppet that has the first Head handle
			//	avoids the need for an extra Head layer param, but also favors the first one
			//	hopefully OK as it's only used a fallback for when eye handles can't be found
		}

		self.headPuppet = newHead;	// used for fallback eyebrow measurement, might be null

		self.aViews = args.getStaticParam("viewLayers");

		self.aViews.forEach(function (view, viewIndex) {
			var featureMap = {};
			headFeatureTagDefinitions.forEach( function (tagDefn) {
				featureMap[tagDefn.id] = false;
			});
			eyeFeatureTagDefinitions.forEach( function (tagDefn) {
				featureMap[tagDefn.id] = false;
			});
			if (includeFaceB) {
				faceFeatureTagDefinitions.forEach( function (tagDefn) {
					featureMap[tagDefn.id] = false;
				});
			}
			if (includePupilsB) {
				pupilFeatureTagDefinitions.forEach( function (tagDefn) {
					featureMap[tagDefn.id] = false;
				});
			}

			self.aFeatureLabels.push(featureMap);
			computeInitTransforms(self, args, view, viewIndex);
			computePuppetMeasurements(self, args, viewIndex, includeFaceB, includePupilsB, includeMouthB);
		});
	}

	function getPauseKeyValue (inputParamId, args) { // method on behavior that is attached to a puppet, only onstage
		return (args.getParamEventValue(inputParamId, cameraInputPauseKeyCodeV2, null, null, true) || 0) && 1;
	}
	
	function onFilterLiveInputs (self, args, filterEyeGazeB) { // method on behavior that is attached to a puppet, only onstage
		var inputParamId = "cameraInput", inputLiveB = args.isParamEventLive(inputParamId), paramOutputKey = args.getParamEventOutputKey(inputParamId);

		if (inputLiveB) {
			var smoothingRate = args.getParam("smoothingRate"), bSnapEyeGaze = false, 
				eyeDartParams = { 
					numEyeDartDirections: 8, 
					minSnapOffsetDist: 0.5, 
					minGazeDuration: 1, 
					leftRightScaleFactor: 0.7, 
					upDownScaleFactor: 0.8, 
					offAxisScaleFactor: 1
				},
				head14 = getHead14(args),
				pauseKeyDown = getPauseKeyValue(inputParamId, args);

			// if filtering eye gaze, get relevant params
			if (filterEyeGazeB) {
				bSnapEyeGaze = args.getParam("snapEyeGaze");
				eyeDartParams.numEyeDartDirections = args.getParam("numEyeDartDirections");
				eyeDartParams.minSnapOffsetDist = args.getParam("minSnapOffsetDist");
				eyeDartParams.minGazeDuration = args.getParam("minGazeDuration");
				eyeDartParams.leftRightScaleFactor = args.getParam("leftRightScaleFactor");
				eyeDartParams.upDownScaleFactor = args.getParam("upDownScaleFactor");
				eyeDartParams.offAxisScaleFactor = args.getParam("offAxisScaleFactor");
			}

			if (bSnapEyeGaze) {
				var currentTime = args.currentTime;
				head14 = snapEyeGaze(self, currentTime, eyeDartParams, head14);
			}
			// when not snapping eye gaze, we need to boost the eye offset strength
			// in order to get full range of pupil motion; it would perhaps be better
			// to perform this adjustment in the original head14 computation ...
			else {
				head14["Head/LeftEyeGazeX"] *= 2.0;
				head14["Head/LeftEyeGazeY"] *= 1.5;
				//		
				head14["Head/RightEyeGazeX"] *= 2.0;
				head14["Head/RightEyeGazeY"] *= 1.5;
			}

			// NOTE: for now, we only apply pose-to-pose filtering on facial animation, and
			// eye gaze filtering is handled separately. but we may want to also apply 
			// pose-to-pose filtering to eye gaze (or other behaviors) if it work well ...
			if (!filterEyeGazeB) {
				if (!self.facePoseToPoseFilter) {
					self.facePoseToPoseFilter = new PoseToPoseFilter();
				}

				updateFacePoseToPoseFilterParams(self, args);

				var enablePoseToPoseFilterB = ((args.getParam("poseFilteringLevel") > 0) ||
												args.getParam("enablePoseToPoseAdvancedParams"));

				if (enablePoseToPoseFilterB) {
					head14 = self.facePoseToPoseFilter.getData(args.currentTime, smoothingRate, head14);			
				}
			}

			// smoothing comes last in the filter chain so that it has an effect in all cases
			head14 = self.faceSmoothingFilter.getData(args.currentTime, smoothingRate, head14, pauseKeyDown);

			for (var label in head14) {
				if (head14.hasOwnProperty(label)) {
					var sustainPreviousValue = !canIterpolateHeadLabel(label) || (label.indexOf("EyeGaze") >= 0 && bSnapEyeGaze && smoothingRate === 0);
					args.eventGraph.publish1D(paramOutputKey + "Filtered/Head/" + label, args.currentTime, 
											  head14[label], sustainPreviousValue);
				}
			}
		}
	}

	return {
		mouthShapeLayerTagDefinitions,
		mouthParentLayerTagDefinition,
		viewLayerTagDefinitions,
		miscLayerTagDefinitions,
		canIterpolateHeadLabel,
		head14Labels,
		eyelidLayerTagDefinitions,
		pupilLayerTagDefinitions,
		headFeatureTagDefinitions,
		faceFeatureTagDefinitions,
		eyeFeatureTagDefinitions,
		pupilFeatureTagDefinitions,
		defineHandleParams,
		applyParamFactorToNamedTransformCustom,
		getPuppetMeasurement,
		addTransformTranslate,
		getPuppetTransforms,
		computePuppetTransforms,
		computePuppetMeasurements,
		makeLayerIdFromLabel,
		makeHandleIdFromLabel,
		getHandle,
		SmoothingFilter,
		getEyeDartDirectionFromAngle,
		snapToPredefinedEyeDartDirection,
		cameraInputPauseKeyCodeV2,	
		cameraInputPauseKeyCodeEnglish,
		cameraInputPauseKeyCodeGerman,
		cameraInputParameterDefinitionV2,
		getHead14,
		getFilteredHead14,
		computeInitTransforms,
		applyPuppetTransforms,
		onCreateStageBehavior,
		onResetRehearsalData,
		getPauseKeyValue,
		onFilterLiveInputs
	};

}); // end define
